/*
 *  $Id: enums.c 28973 2025-12-10 18:45:40Z yeti-dn $
 *  Copyright (C) 2005-2021 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <libgwyddion/gwyddion.h>

/**
 * GwyMergeType:
 * @GWY_MERGE_UNION: Union (logical or) merging.
 * @GWY_MERGE_INTERSECTION: Intersection (logical and) merging.
 *
 * Mask merge type (namely used in grain processing).
 **/
/**
 * gwy_merge_type_get_enum:
 *
 * Returns #GwyEnum for #GwyMergeType enum type.
 *
 * Returns: %NULL-terminated #GwyEnum which must not be modified nor freed.
 **/
const GwyEnum*
gwy_merge_type_get_enum(void)
{
    static const GwyEnum entries[] = {
        { N_("Union"),            GWY_MERGE_UNION,        },
        { N_("Intersection"),     GWY_MERGE_INTERSECTION, },
        { NULL,                   0,                      },
    };
    return entries;
}

/**
 * GwyInterpolationType:
 * @GWY_INTERPOLATION_ROUND: Round interpolation (more precisely symmetric nearest neighbour interpolation).
 * @GWY_INTERPOLATION_LINEAR: Linear interpolation.
 * @GWY_INTERPOLATION_KEY: Cubic Key's interpolation (with a=-1/2).
 * @GWY_INTERPOLATION_BSPLINE: B-spline interpolation.
 * @GWY_INTERPOLATION_OMOMS: Omoms interpolation.
 * @GWY_INTERPOLATION_NNA: Nearest neighbour approximation.
 * @GWY_INTERPOLATION_SCHAUM: Cubic Schaum interpolation.
 *
 * Interpolation types.
 **/
/**
 * gwy_interpolation_type_get_enum:
 *
 * Returns #GwyEnum for #GwyInterpolationType enum type.
 *
 * Returns: %NULL-terminated #GwyEnum which must not be modified nor freed.
 **/
const GwyEnum*
gwy_interpolation_type_get_enum(void)
{
    static const GwyEnum entries[] = {
        /* TRANSLATORS: Interpolation type (AKA nearest neighbour) */
        { N_("Round"),   GWY_INTERPOLATION_ROUND,   },
        { N_("Linear"),  GWY_INTERPOLATION_LINEAR,  },
        { N_("Key"),     GWY_INTERPOLATION_KEY,     },
        { N_("BSpline"), GWY_INTERPOLATION_BSPLINE, },
        { N_("OMOMS"),   GWY_INTERPOLATION_OMOMS,   },
        { N_("NNA"),     GWY_INTERPOLATION_NNA,     },
        { N_("Schaum"),  GWY_INTERPOLATION_SCHAUM,  },
        { NULL,          0,                         },
    };
    return entries;
}

/**
 * GwyWindowingType:
 * @GWY_WINDOWING_NONE: No windowing is applied.
 * @GWY_WINDOWING_HANN: Hann window.
 * @GWY_WINDOWING_HAMMING: Hamming window.
 * @GWY_WINDOWING_BLACKMANN: Blackmann window.
 * @GWY_WINDOWING_LANCZOS: Lanczos window.
 * @GWY_WINDOWING_WELCH: Welch window.
 * @GWY_WINDOWING_RECT: Rectangular window.
 * @GWY_WINDOWING_NUTTALL: Nuttall window.
 * @GWY_WINDOWING_FLAT_TOP: Flat-top window.
 * @GWY_WINDOWING_KAISER25: Kaiser window with β=2.5.
 *
 * Frequency windowing type.
 **/
/**
 * gwy_windowing_type_get_enum:
 *
 * Returns #GwyEnum for #GwyWindowingType enum type.
 *
 * Returns: %NULL-terminated #GwyEnum which must not be modified nor freed.
 **/
const GwyEnum*
gwy_windowing_type_get_enum(void)
{
    static const GwyEnum entries[] = {
        { gwy_NC("windowing", "None"), GWY_WINDOWING_NONE,      },
        { N_("Hann"),                  GWY_WINDOWING_HANN,      },
        { N_("Hamming"),               GWY_WINDOWING_HAMMING,   },
        { N_("Blackmann"),             GWY_WINDOWING_BLACKMANN, },
        { N_("Lanczos"),               GWY_WINDOWING_LANCZOS,   },
        { N_("Welch"),                 GWY_WINDOWING_WELCH,     },
        { N_("Rect"),                  GWY_WINDOWING_RECT,      },
        { N_("Nuttall"),               GWY_WINDOWING_NUTTALL,   },
        { N_("Flat-top"),              GWY_WINDOWING_FLAT_TOP,  },
        { N_("Kaiser 2.5"),            GWY_WINDOWING_KAISER25,  },
        { NULL,                        0,                       },
    };
    return entries;
}

/**
 * gwy_correlation_type_get_enum:
 *
 * Returns #GwyEnum for #GwyCorrelationType enum type.
 *
 * Returns: %NULL-terminated #GwyEnum which must not be modified nor freed.
 **/
const GwyEnum*
gwy_correlation_type_get_enum(void)
{
    static const GwyEnum entries[] = {
        { gwy_NC("correlation", "Normal"), GWY_CORRELATION_NORMAL, },
        { N_("FFT"),                       GWY_CORRELATION_FFT,    },
        { N_("Phase only"),                GWY_CORRELATION_POC,    },
        { NULL,                            0,                      },
    };
    return entries;
}

/**
 * GwyMaskingType:
 * @GWY_MASK_EXCLUDE: Exclude data under mask, i.e. take into account only
 *                    data not covered by the mask.
 * @GWY_MASK_INCLUDE: Take into account only data under the mask.
 * @GWY_MASK_IGNORE: Ignore mask, if present, and use all data.
 *
 * Mask handling in procedures that can apply masking.
 *
 * Note at present many procedures do not have a masking argument and hence cannot apply masks in exclude mode.
 **/
/**
 * gwy_masking_type_get_enum:
 *
 * Returns #GwyEnum for #GwyMaskingType enum type.
 *
 * Returns: %NULL-terminated #GwyEnum which must not be modified nor freed.
 **/
const GwyEnum*
gwy_masking_type_get_enum(void)
{
    static const GwyEnum entries[] = {
        { N_("Exclude masked region"),          GWY_MASK_EXCLUDE, },
        { N_("Include only masked region"),     GWY_MASK_INCLUDE, },
        { N_("Use entire image (ignore mask)"), GWY_MASK_IGNORE,  },
        { NULL,                                 0,                },
    };
    return entries;
}

/**
 * GwyOrientation:
 * @GWY_ORIENTATION_HORIZONTAL: Horizontal orientation (row-wise in images).
 * @GWY_ORIENTATION_VERTICAL: Vertical orientation (column-wise in images).
 *
 * Orientation type.
 *
 * The enum is identical to #GtkOrientation and the values should be interchangeable. It is, however, available in
 * non-GUI context.
 *
 * See also #GwyDirectionFlags for flags, if the directions are not exclusive (and possibly ‘none’ is also an option).
 **/
/**
 * gwy_orientation_get_enum:
 *
 * Returns #GwyEnum for #GwyOrientation enum type.
 *
 * Returns: %NULL-terminated #GwyEnum which must not be modified nor freed.
 **/
const GwyEnum*
gwy_orientation_get_enum(void)
{
    static const GwyEnum entries[] = {
        { gwy_NC("orientation", "Horizontal"), GWY_ORIENTATION_HORIZONTAL, },
        { gwy_NC("orientation", "Vertical"),   GWY_ORIENTATION_VERTICAL,   },
        { NULL,                                0,                          },
    };
    return entries;
}

/**
 * GwyDirectionFlags:
 * @GWY_DIRECTION_HORIZONTAL: Horizontal direction (row-wise in images).
 * @GWY_DIRECTION_VERTICAL: Vertical orientation (column-wise in images).
 * @GWY_DIRECTION_BOTH: Both directions.
 *
 * Orientation flags.
 *
 * If needed, zero (no flag set) can be used to represent no direction.
 *
 * See also #GwyOrientation for the exclusive variant (the orientation is always exactly one of the two).
 **/
/**
 * gwy_direction_flags_get_enum:
 *
 * Returns #GwyEnum for #GwyDirectionFlags enum type.
 *
 * Returns: %NULL-terminated #GwyEnum which must not be modified nor freed.
 **/
const GwyEnum*
gwy_direction_flags_get_enum(void)
{
    static const GwyEnum entries[] = {
        { gwy_NC("orientaton", "Horizontal"), GWY_DIRECTION_HORIZONTAL, },
        { gwy_NC("orientaton", "Vertical"),   GWY_DIRECTION_VERTICAL,   },
        { gwy_NC("orientaton", "Both"),       GWY_DIRECTION_BOTH,       },
        { NULL,                               0,                        },
    };
    return entries;
}

/**
 * GwySignFlags:
 * @GWY_SIGN_NEGATIVE: Negative sign (upwards).
 * @GWY_SIGN_POSITIVE: Positive sign (downwards).
 * @GWY_SIGN_BOTH: Both signs.
 *
 * Sign flags.
 *
 * If needed, zero (no flag set) can be used to represent neither sign.
 **/
/**
 * gwy_sign_flags_get_enum:
 *
 * Returns #GwyEnum for #GwySignFlags enum type.
 *
 * Returns: %NULL-terminated #GwyEnum which must not be modified nor freed.
 **/
const GwyEnum*
gwy_sign_flags_get_enum(void)
{
    static const GwyEnum entries[] = {
        { gwy_NC("sign", "Negative"), GWY_SIGN_NEGATIVE, },
        { gwy_NC("sign", "Positive"), GWY_SIGN_POSITIVE, },
        { gwy_NC("sign", "Both"),     GWY_SIGN_BOTH,     },
        { NULL,                       0,                 },
    };
    return entries;
}


/**
 * SECTION: enums
 * @title: Enums
 * @short_description: Common enumerations
 **/

/**
 * GwyPlaneFitQuantity:
 * @GWY_PLANE_FIT_A: Constant coefficient (mean value).
 * @GWY_PLANE_FIT_BX: Linear coefficient in @x, where @x is in pixel coordinates.
 * @GWY_PLANE_FIT_BY: Linear coefficient in @y, where @y is in pixel coordinates.
 * @GWY_PLANE_FIT_ANGLE: Slope orientation in (x,y) plane (in radians).
 * @GWY_PLANE_FIT_SLOPE: Absolute slope value (that is sqrt(bx*bx + by*by)).
 * @GWY_PLANE_FIT_S0: Residual sum of squares.
 * @GWY_PLANE_FIT_S0_REDUCED: Slope-reduced residual sum of squares.
 *
 * Local plane fitting quantity to request from gwy_field_area_fit_local_planes() and similar functions.
 **/

/**
 * GwyFieldCached:
 * @GWY_FIELD_CACHE_MIN: Overall minimum.
 * @GWY_FIELD_CACHE_MAX: Overall maximum.
 * @GWY_FIELD_CACHE_SUM: Sum of all values.
 * @GWY_FIELD_CACHE_RMS: Root mean square.
 * @GWY_FIELD_CACHE_MED: Median.
 * @GWY_FIELD_CACHE_ARF: Auto-range from.
 * @GWY_FIELD_CACHE_ART: Auto-range to.
 * @GWY_FIELD_CACHE_ARE: Surface area.
 * @GWY_FIELD_CACHE_VAR: Variation.
 * @GWY_FIELD_CACHE_ENT: Entropy.
 * @GWY_FIELD_CACHE_MSQ: Mean square.
 * @GWY_FIELD_CACHE_SIZE: The size of statistics cache.
 *
 * Cached data field quantity type.
 *
 * There should be little need to this enum directly except in libgwyprocess
 * methods.
 **/

/**
 * GwyExteriorType:
 * @GWY_EXTERIOR_UNDEFINED: The values corresponding to or calculated from exterior data values are undefined, they
 *                          may be left unset or set to bogus values.  The caller must handle them itself afterwards,
 *                          for instance by resizing the result to consist of valid data only.
 * @GWY_EXTERIOR_BORDER: Values of exterior pixels are considered to be equal to the values of the nearest
 *                              interior pixels.
 * @GWY_EXTERIOR_MIRROR: The data is considered to be periodically repeated, with odd instances reflected (the
 *                              total period is thus twice the size of the data).
 * @GWY_EXTERIOR_PERIODIC: The data is considered to be periodically repeated.
 * @GWY_EXTERIOR_FIXED_VALUE: Values of exterior pixels are considered to be all equal to a user-specified value.
 * @GWY_EXTERIOR_LAPLACE: Values of exterior pixels are extended using Laplace interpolation (actually extrapolation)
 *                        like gwy_field_laplace_solve().  Only some functions implement this method.
 *
 * Methods to handle pixels outside data.
 *
 * Many methods currently use a fixed metod of handling of exterior pixels, for example area calculation uses
 * extension (border and mirror coincide), convolution uses mirror extension, rotation fills exterior with a fixed
 * value.
 **/

/**
 * GwyRotateResizeType:
 * @GWY_ROTATE_RESIZE_SAME_SIZE: The result has the same area as the original.
 * @GWY_ROTATE_RESIZE_EXPAND: The result is sufficiently large so that all original data are present (usually meaning
 *                            that there will be also lot of exterior).
 * @GWY_ROTATE_RESIZE_CUT: The result is optimally cut to interior data only.
 *
 * Type of rotated data field size determination method.
 **/

/**
 * GwyComputationStateType:
 * @GWY_COMPUTATION_STATE_INIT: Iterator was set up, the next step will actually create temporary data structures and
 *                              precalculate values.
 * @GWY_COMPUTATION_STATE_ITERATE: Iteration is in progress, the @fraction field of state struct holds the fraction
 *                                 completed.
 * @GWY_COMPUTATION_STATE_FINISHED: Calculation has been finished, further calls to the iterator will be no-op.
 *
 * Common iterative computation iterator state type.
 **/

/**
 * GwyComputationState:
 * @state: Current computation state, usually of #GwyComputationStateType, but particular iterators can define their
 *         own types.
 * @fraction: Fraction of computation completed.  For staged algorithms, the fraction refers to the current stage
 *            only.
 *
 * State of iterative computation.
 *
 * Iterators usually append their own private state data, therefore it must not be assumed the public fields @state
 * and @fraction are the only fields.
 *
 * A typical iteration, assuming an iterative computation `foo' with the default #GwyComputationStateType state could
 * be:
 * <informalexample><programlisting>
 * GwyComputationStateType *state;
 * <!-- Hello, gtk-doc! -->
 * state = gwy_field_foo_init(GwyField *field, ...);
 * do {
 *     gwy_field_foo_iteration(state);
 *     /<!-- -->* Update progress using state->fraction,
 *         let Gtk+ main loop run, check for abort, ... *<!-- -->/
 *     if (aborted) {
 *         gwy_field_foo_finalize(state);
 *         return FALSE;
 *     }
 * } while (state->state != GWY_COMPUTATION_STATE_FINISHED);
 * gwy_field_foo_finalize(state);
 * return TRUE;
 * </programlisting></informalexample>
 **/
GType
gwy_computation_state_get_type(void)
{
    static GType compstate_type = 0;

    if (G_UNLIKELY(!compstate_type))
        compstate_type = g_pointer_type_register_static("GwyComputationState");

    return compstate_type;
}

/**
 * gwy_computation_state_get_state:
 * @compstate: Computation state.
 *
 * Gets the state field of a computation state struct.
 *
 * Useful mostly for language bindings.
 *
 * Returns: The state field of @compstate.
 **/
gint
gwy_computation_state_get_state(GwyComputationState *compstate)
{
    return compstate->state;
}

/**
 * gwy_computation_state_get_fraction:
 * @compstate: Computation state.
 *
 * Gets the fraction field of a computation state struct.
 *
 * Returns: The fraction field of @compstate.
 **/
gdouble
gwy_computation_state_get_fraction(GwyComputationState *compstate)
{
    return compstate->fraction;
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
