/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2026 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - MassXpert, model polymer chemistries and simulate mass spectrometric data;
 * - MineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Qt includes
#include <QDebug>
#include <QStringList>
#include <QList>
#include <QString>

/////////////////////// Local includes
#include "ConfigSettings.hpp"


namespace MsXpS
{
namespace MassXpert
{


ConfigSettings::ConfigSettings(const QString &application_name)
  : m_applicationName{application_name}
{
}

ConfigSettings::ConfigSettings(const ConfigSettings &other)
  : m_applicationName(other.m_applicationName)
{
  for(const ConfigSettingSPtr &config_setting_sp : other.m_configSettings)
    m_configSettings.push_back(
      std::make_shared<ConfigSetting>(*config_setting_sp));
}

ConfigSettings &
ConfigSettings::operator=(const ConfigSettings &other)
{
  if(&other == this)
    return *this;

  m_applicationName = other.m_applicationName;

  for(const ConfigSettingSPtr &config_setting_sp : other.m_configSettings)
    m_configSettings.push_back(
      std::make_shared<ConfigSetting>(*config_setting_sp));

  return *this;
}

ConfigSettings::~ConfigSettings()
{
}

void
ConfigSettings::setApplicationName(const QString &application_name)
{
  m_applicationName = application_name;
}

QString
ConfigSettings::getApplicationName()
{
  return m_applicationName;
}

QString
ConfigSettings::getConfigSettingKey(
  const ConfigSettingSPtr &config_setting_sp) const
{
  std::vector<ConfigSettingSPtr>::const_iterator the_iterator_cst =
    std::find_if(
      m_configSettings.cbegin(),
      m_configSettings.cend(),
      [config_setting_sp](const ConfigSettingSPtr &iter_config_setting_sp) {
        return iter_config_setting_sp == config_setting_sp;
      });

  if(the_iterator_cst != m_configSettings.cend())
    return (*the_iterator_cst)->m_key;

  return QString();
}


const ConfigSettingSPtr
ConfigSettings::getConfigSetting(QString key, UserType user_type) const
{
  std::vector<ConfigSettingSPtr>::const_iterator the_iterator_cst =
    std::find_if(
      m_configSettings.cbegin(),
      m_configSettings.cend(),
      [key, user_type](const ConfigSettingSPtr &iter_config_setting_sp) {
        return (iter_config_setting_sp->m_key == key &&
                iter_config_setting_sp->m_userType == user_type);
      });

  if(the_iterator_cst != m_configSettings.cend())
    return *the_iterator_cst;

  return nullptr;
}

std::vector<ConfigSettingSPtr> &
ConfigSettings::getConfigSettingsRef()
{
  return m_configSettings;
}

const std::vector<ConfigSettingSPtr> &
ConfigSettings::getConfigSettingsCstRef() const
{
  return m_configSettings;
}

QString
ConfigSettings::toString() const
{
  QString text;

  for(const ConfigSettingSPtr &config_setting_sp : m_configSettings)
    text += config_setting_sp->toString();

  return text;
}


} // namespace MassXpert
} // namespace MsXpS
